#!/bin/bash

# Auth - Login
cat > auth/login.php << 'EOF'
<?php
require_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$data = json_decode(file_get_contents("php://input"));

if (!empty($data->email) && !empty($data->password)) {
    $query = "SELECT id, name, email, password FROM users WHERE email = :email LIMIT 1";
    $stmt = $db->prepare($query);
    $stmt->bindParam(":email", $data->email);
    $stmt->execute();

    if ($stmt->rowCount() > 0) {
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (password_verify($data->password, $row['password'])) {
            $token = bin2hex(random_bytes(32));
            
            echo json_encode([
                "success" => true,
                "message" => "Login berhasil",
                "data" => [
                    "id" => $row['id'],
                    "name" => $row['name'],
                    "email" => $row['email'],
                    "token" => $token
                ]
            ]);
        } else {
            echo json_encode([
                "success" => false,
                "message" => "Password salah"
            ]);
        }
    } else {
        echo json_encode([
            "success" => false,
            "message" => "Email tidak ditemukan"
        ]);
    }
} else {
    echo json_encode([
        "success" => false,
        "message" => "Email dan password harus diisi"
    ]);
}
?>
EOF

# Auth - Register
cat > auth/register.php << 'EOF'
<?php
require_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$data = json_decode(file_get_contents("php://input"));

if (!empty($data->name) && !empty($data->email) && !empty($data->password)) {
    $check_query = "SELECT id FROM users WHERE email = :email";
    $check_stmt = $db->prepare($check_query);
    $check_stmt->bindParam(":email", $data->email);
    $check_stmt->execute();

    if ($check_stmt->rowCount() > 0) {
        echo json_encode([
            "success" => false,
            "message" => "Email sudah terdaftar"
        ]);
    } else {
        $query = "INSERT INTO users (name, email, password) VALUES (:name, :email, :password)";
        $stmt = $db->prepare($query);
        
        $stmt->bindParam(":name", $data->name);
        $stmt->bindParam(":email", $data->email);
        $hashed_password = password_hash($data->password, PASSWORD_DEFAULT);
        $stmt->bindParam(":password", $hashed_password);

        if ($stmt->execute()) {
            echo json_encode([
                "success" => true,
                "message" => "Registrasi berhasil"
            ]);
        } else {
            echo json_encode([
                "success" => false,
                "message" => "Registrasi gagal"
            ]);
        }
    }
} else {
    echo json_encode([
        "success" => false,
        "message" => "Semua field harus diisi"
    ]);
}
?>
EOF

# Transactions - Get
cat > transactions/get.php << 'EOF'
<?php
require_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$user_id = isset($_GET['user_id']) ? $_GET['user_id'] : die(json_encode(["success" => false, "message" => "User ID required"]));

$query = "SELECT * FROM transactions WHERE user_id = :user_id";

if (isset($_GET['start_date']) && isset($_GET['end_date'])) {
    $query .= " AND date BETWEEN :start_date AND :end_date";
}

$query .= " ORDER BY date DESC, id DESC";

$stmt = $db->prepare($query);
$stmt->bindParam(":user_id", $user_id);

if (isset($_GET['start_date']) && isset($_GET['end_date'])) {
    $stmt->bindParam(":start_date", $_GET['start_date']);
    $stmt->bindParam(":end_date", $_GET['end_date']);
}

$stmt->execute();

$transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

echo json_encode([
    "success" => true,
    "data" => $transactions
]);
?>
EOF

# Transactions - Add
cat > transactions/add.php << 'EOF'
<?php
require_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$data = json_decode(file_get_contents("php://input"));

if (!empty($data->user_id) && !empty($data->type) && !empty($data->amount) && !empty($data->category)) {
    $query = "INSERT INTO transactions (user_id, type, amount, category, description, date) 
              VALUES (:user_id, :type, :amount, :category, :description, :date)";
    
    $stmt = $db->prepare($query);
    
    $stmt->bindParam(":user_id", $data->user_id);
    $stmt->bindParam(":type", $data->type);
    $stmt->bindParam(":amount", $data->amount);
    $stmt->bindParam(":category", $data->category);
    $stmt->bindParam(":description", $data->description);
    $date = isset($data->date) ? date('Y-m-d', strtotime($data->date)) : date('Y-m-d');
    $stmt->bindParam(":date", $date);

    if ($stmt->execute()) {
        echo json_encode([
            "success" => true,
            "message" => "Transaksi berhasil ditambahkan",
            "id" => $db->lastInsertId()
        ]);
    } else {
        echo json_encode([
            "success" => false,
            "message" => "Gagal menambahkan transaksi"
        ]);
    }
} else {
    echo json_encode([
        "success" => false,
        "message" => "Data tidak lengkap"
    ]);
}
?>
EOF

# Transactions - Update
cat > transactions/update.php << 'EOF'
<?php
require_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$data = json_decode(file_get_contents("php://input"));

if (!empty($data->id)) {
    $query = "UPDATE transactions SET 
              type = :type,
              amount = :amount,
              category = :category,
              description = :description,
              date = :date
              WHERE id = :id AND user_id = :user_id";
    
    $stmt = $db->prepare($query);
    
    $stmt->bindParam(":id", $data->id);
    $stmt->bindParam(":user_id", $data->user_id);
    $stmt->bindParam(":type", $data->type);
    $stmt->bindParam(":amount", $data->amount);
    $stmt->bindParam(":category", $data->category);
    $stmt->bindParam(":description", $data->description);
    $date = date('Y-m-d', strtotime($data->date));
    $stmt->bindParam(":date", $date);

    if ($stmt->execute()) {
        echo json_encode([
            "success" => true,
            "message" => "Transaksi berhasil diupdate"
        ]);
    } else {
        echo json_encode([
            "success" => false,
            "message" => "Gagal mengupdate transaksi"
        ]);
    }
} else {
    echo json_encode([
        "success" => false,
        "message" => "ID transaksi tidak ditemukan"
    ]);
}
?>
EOF

# Transactions - Delete
cat > transactions/delete.php << 'EOF'
<?php
require_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$id = isset($_GET['id']) ? $_GET['id'] : die(json_encode(["success" => false, "message" => "ID required"]));

$query = "DELETE FROM transactions WHERE id = :id";
$stmt = $db->prepare($query);
$stmt->bindParam(":id", $id);

if ($stmt->execute()) {
    echo json_encode([
        "success" => true,
        "message" => "Transaksi berhasil dihapus"
    ]);
} else {
    echo json_encode([
        "success" => false,
        "message" => "Gagal menghapus transaksi"
    ]);
}
?>
EOF

# Budgets - Get
cat > budgets/get.php << 'EOF'
<?php
require_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$user_id = isset($_GET['user_id']) ? $_GET['user_id'] : die(json_encode(["success" => false, "message" => "User ID required"]));
$month = isset($_GET['month']) ? $_GET['month'] : date('Y-m');

$query = "SELECT * FROM budgets WHERE user_id = :user_id AND month = :month ORDER BY category";

$stmt = $db->prepare($query);
$stmt->bindParam(":user_id", $user_id);
$stmt->bindParam(":month", $month);

$stmt->execute();

$budgets = $stmt->fetchAll(PDO::FETCH_ASSOC);

echo json_encode([
    "success" => true,
    "data" => $budgets
]);
?>
EOF

# Budgets - Add
cat > budgets/add.php << 'EOF'
<?php
require_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$data = json_decode(file_get_contents("php://input"));

if (!empty($data->user_id) && !empty($data->category) && !empty($data->planned_amount) && !empty($data->month)) {
    $query = "INSERT INTO budgets (user_id, category, planned_amount, month) 
              VALUES (:user_id, :category, :planned_amount, :month)
              ON DUPLICATE KEY UPDATE planned_amount = :planned_amount2";
    
    $stmt = $db->prepare($query);
    
    $stmt->bindParam(":user_id", $data->user_id);
    $stmt->bindParam(":category", $data->category);
    $stmt->bindParam(":planned_amount", $data->planned_amount);
    $stmt->bindParam(":planned_amount2", $data->planned_amount);
    $stmt->bindParam(":month", $data->month);

    if ($stmt->execute()) {
        echo json_encode([
            "success" => true,
            "message" => "Budget berhasil ditambahkan"
        ]);
    } else {
        echo json_encode([
            "success" => false,
            "message" => "Gagal menambahkan budget"
        ]);
    }
} else {
    echo json_encode([
        "success" => false,
        "message" => "Data tidak lengkap"
    ]);
}
?>
EOF

# Budgets - Update
cat > budgets/update.php << 'EOF'
<?php
require_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$data = json_decode(file_get_contents("php://input"));

if (!empty($data->id)) {
    $query = "UPDATE budgets SET 
              category = :category,
              planned_amount = :planned_amount,
              month = :month
              WHERE id = :id AND user_id = :user_id";
    
    $stmt = $db->prepare($query);
    
    $stmt->bindParam(":id", $data->id);
    $stmt->bindParam(":user_id", $data->user_id);
    $stmt->bindParam(":category", $data->category);
    $stmt->bindParam(":planned_amount", $data->planned_amount);
    $stmt->bindParam(":month", $data->month);

    if ($stmt->execute()) {
        echo json_encode([
            "success" => true,
            "message" => "Budget berhasil diupdate"
        ]);
    } else {
        echo json_encode([
            "success" => false,
            "message" => "Gagal mengupdate budget"
        ]);
    }
} else {
    echo json_encode([
        "success" => false,
        "message" => "ID budget tidak ditemukan"
    ]);
}
?>
EOF

# Statistics - Monthly
cat > statistics/monthly.php << 'EOF'
<?php
require_once '../config/database.php';

$database = new Database();
$db = $database->getConnection();

$user_id = isset($_GET['user_id']) ? $_GET['user_id'] : die(json_encode(["success" => false, "message" => "User ID required"]));
$start_month = isset($_GET['start_month']) ? $_GET['start_month'] : date('Y-m', strtotime('-5 months'));
$end_month = isset($_GET['end_month']) ? $_GET['end_month'] : date('Y-m');

$query = "SELECT 
            DATE_FORMAT(date, '%Y-%m') as month,
            SUM(CASE WHEN type = 'income' THEN amount ELSE 0 END) as income,
            SUM(CASE WHEN type = 'expense' THEN amount ELSE 0 END) as expense
          FROM transactions 
          WHERE user_id = :user_id 
          AND DATE_FORMAT(date, '%Y-%m') BETWEEN :start_month AND :end_month
          GROUP BY DATE_FORMAT(date, '%Y-%m')
          ORDER BY month";

$stmt = $db->prepare($query);
$stmt->bindParam(":user_id", $user_id);
$stmt->bindParam(":start_month", $start_month);
$stmt->bindParam(":end_month", $end_month);

$stmt->execute();

$stats = $stmt->fetchAll(PDO::FETCH_ASSOC);

echo json_encode([
    "success" => true,
    "data" => $stats
]);
?>
EOF

echo "All API files created successfully!"
